bl_info = {
    "name": "Voxel Generator v4.3.1 (Uniform Grid Fill - Color Bleed Fix)",
    "author": "You + ChatGPT",
    "version": (4, 3, 1),
    "blender": (3, 0, 0),
    "location": "View3D > Sidebar > Voxel Tool",
    "description": "Uniform voxel grid fill (no giant cubes). MeshSafe BVH + MeshSafeColorSampler. FIX: more stable color on tiny details (6-dir raycast + bilinear texture sampling).",
    "category": "Object",
}

import bpy, os, math, random, traceback, bmesh
import mathutils
from mathutils.bvhtree import BVHTree

# ------------------------------------------------------------
# UI / Logging
# ------------------------------------------------------------
def popup(lines, title="Voxel Generator", icon='INFO'):
    def _draw(self, context):
        for ln in str(lines).splitlines():
            self.layout.label(text=ln)
    try:
        bpy.context.window_manager.popup_menu(_draw, title=title, icon=icon)
    except Exception:
        pass

def log_to_text(msg, txt_name="VOXEL_RESULT.txt"):
    print(msg)
    txt = bpy.data.texts.get(txt_name) or bpy.data.texts.new(txt_name)
    try:
        txt.clear()
    except Exception:
        pass
    txt.write(str(msg) + "\n")

# ------------------------------------------------------------
# Small utils
# ------------------------------------------------------------
def clamp01(x): return 0.0 if x < 0.0 else (1.0 if x > 1.0 else x)

def linear_to_srgb_1(c):
    return 12.92*c if c <= 0.0031308 else 1.055*(c**(1/2.4)) - 0.055

def rgb_to_hex(rgb_lin):
    r = int(round(clamp01(linear_to_srgb_1(rgb_lin[0])) * 255))
    g = int(round(clamp01(linear_to_srgb_1(rgb_lin[1])) * 255))
    b = int(round(clamp01(linear_to_srgb_1(rgb_lin[2])) * 255))
    return f"#{r:02X}{g:02X}{b:02X}"

def ensure_collection(name):
    col = bpy.data.collections.get(name)
    if not col:
        col = bpy.data.collections.new(name)
        bpy.context.scene.collection.children.link(col)
    return col

def ensure_child_collection(parent_name, child_name):
    parent = ensure_collection(parent_name)
    child = bpy.data.collections.get(child_name)
    if not child:
        child = bpy.data.collections.new(child_name)
        bpy.context.scene.collection.children.link(child)
    if child.name not in [c.name for c in parent.children]:
        try: parent.children.link(child)
        except Exception: pass
    return child

def clear_collection_objects(col):
    for obj in list(col.objects):
        bpy.data.objects.remove(obj, do_unlink=True)

def move_to_collection(obj, col):
    for c in list(obj.users_collection):
        try: c.objects.unlink(obj)
        except Exception: pass
    col.objects.link(obj)

def world_bbox(obj):
    mw = obj.matrix_world
    pts = [mw @ mathutils.Vector(c) for c in obj.bound_box]
    xs=[p.x for p in pts]; ys=[p.y for p in pts]; zs=[p.z for p in pts]
    return (min(xs), min(ys), min(zs), max(xs), max(ys), max(zs))

def ensure_cube_mesh(name="VoxelCubeMeshBase"):
    me = bpy.data.meshes.get(name)
    if me and len(me.vertices) >= 8 and len(me.polygons) >= 6:
        return me
    if me:
        try: bpy.data.meshes.remove(me)
        except Exception: pass
    me = bpy.data.meshes.new(name)
    bm = bmesh.new()
    bmesh.ops.create_cube(bm, size=1.0)
    bm.to_mesh(me)
    bm.free()
    me.update()
    return me

# ------------------------------------------------------------
# Import / Join
# ------------------------------------------------------------
def import_any(path):
    path = os.path.abspath(bpy.path.abspath(path))
    if not os.path.isfile(path):
        raise FileNotFoundError(f"Không tìm thấy file: {path}")
    ext = os.path.splitext(path)[1].lower()
    bpy.ops.object.select_all(action='DESELECT')
    if ext == ".obj":
        bpy.ops.import_scene.obj(filepath=path)
    elif ext == ".fbx":
        bpy.ops.import_scene.fbx(filepath=path)
    elif ext in (".glb", ".gltf"):
        bpy.ops.import_scene.gltf(filepath=path)
    else:
        raise ValueError("File không hỗ trợ. Dùng .obj / .fbx / .glb / .gltf")
    sel = [o for o in bpy.context.selected_objects if o.type == 'MESH']
    if not sel:
        sel = [o for o in bpy.context.scene.objects if o.type == 'MESH' and not o.library]
    if not sel:
        raise RuntimeError("Không tìm thấy mesh nào sau khi import.")
    return sel

def join_meshes(objs):
    for o in objs:
        bpy.ops.object.select_all(action='DESELECT')
        o.select_set(True)
        bpy.context.view_layer.objects.active = o
        bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)
    bpy.ops.object.select_all(action='DESELECT')
    for o in objs:
        o.select_set(True)
    bpy.context.view_layer.objects.active = objs[0]
    if len(objs) > 1:
        bpy.ops.object.join()
    merged = bpy.context.view_layer.objects.active
    merged.name = "Voxel_Source_Merged"
    return merged

# ------------------------------------------------------------
# MeshSafe BVH for inside test (no mesh lifetime dependency)
# ------------------------------------------------------------
def build_meshsafe_bvh(obj, deps):
    eval_obj = obj.evaluated_get(deps)
    me = eval_obj.to_mesh()
    try:
        if not me.loop_triangles:
            me.calc_loop_triangles()

        mw = eval_obj.matrix_world
        verts_w = [mw @ v.co for v in me.vertices]
        tris = [tuple(tri.vertices) for tri in me.loop_triangles]

        bvh = BVHTree.FromPolygons(verts_w, tris, all_triangles=True)
        if bvh is None:
            raise RuntimeError("Không tạo được BVH (FromPolygons).")

        minx, miny, minz, maxx, maxy, maxz = world_bbox(obj)
        sx = maxx-minx; sy=maxy-miny; sz=maxz-minz
        diag = math.sqrt(sx*sx + sy*sy + sz*sz) + 1e-9
        return bvh, diag
    finally:
        try: eval_obj.to_mesh_clear()
        except Exception: pass

def _ray_parity(bvh, point, direction, max_dist, eps=1e-5, max_hits=256):
    count = 0
    o = point.copy()
    remaining = max_dist
    for _ in range(max_hits):
        loc, normal, idx, dist = bvh.ray_cast(o, direction, remaining)
        if loc is None:
            break
        count += 1
        stepd = dist + eps
        o = o + direction * stepd
        remaining -= stepd
        if remaining <= eps:
            break
    return (count % 2) == 1

def inside_vote(bvh, point, diag):
    max_dist = diag * 2.5
    axes = (mathutils.Vector((1,0,0)), mathutils.Vector((0,1,0)), mathutils.Vector((0,0,1)))
    votes = 0
    for a in axes:
        inside_pos = _ray_parity(bvh, point - a*1e-5, a, max_dist)
        inside_neg = _ray_parity(bvh, point + a*1e-5, -a, max_dist)
        if inside_pos or inside_neg:
            votes += 1
    return votes >= 2

# ------------------------------------------------------------
# Palette (k-means)
# ------------------------------------------------------------
def dist2(a, b):
    dx=a[0]-b[0]; dy=a[1]-b[1]; dz=a[2]-b[2]
    return dx*dx + dy*dy + dz*dz

def kmeans_pp_init(colors, k, seed=42):
    rng = random.Random(int(seed))
    centers = [colors[rng.randrange(len(colors))]]
    dists = [dist2(c, centers[0]) for c in colors]
    for _ in range(1, k):
        total = sum(dists) or 1.0
        r = rng.random() * total
        s = 0.0; idx = 0
        for i, d in enumerate(dists):
            s += d
            if s >= r:
                idx = i; break
        centers.append(colors[idx])
        last = centers[-1]
        for i, c in enumerate(colors):
            d = dist2(c, last)
            if d < dists[i]:
                dists[i] = d
    return centers

def kmeans_palette(colors, k, iters=16, seed=42):
    k = max(1, min(int(k), len(colors)))
    centers = kmeans_pp_init(colors, k, seed)
    for _ in range(iters):
        buckets = [[] for _ in range(k)]
        for c in colors:
            j = min(range(k), key=lambda t: dist2(c, centers[t]))
            buckets[j].append(c)
        moved = 0.0
        for j in range(k):
            if buckets[j]:
                mx = sum(c[0] for c in buckets[j]) / len(buckets[j])
                my = sum(c[1] for c in buckets[j]) / len(buckets[j])
                mz = sum(c[2] for c in buckets[j]) / len(buckets[j])
                newc = (clamp01(mx), clamp01(my), clamp01(mz))
            else:
                newc = colors[random.randrange(len(colors))]
            moved += dist2(newc, centers[j])
            centers[j] = newc
        if moved < 1e-8:
            break
    return centers

# ------------------------------------------------------------
# Materials / Mesh per color
# ------------------------------------------------------------
def make_color_material(rgb_lin, cache):
    key = (round(rgb_lin[0],4), round(rgb_lin[1],4), round(rgb_lin[2],4))
    if key in cache:
        return cache[key]
    hx = rgb_to_hex(rgb_lin)
    m = bpy.data.materials.new(f"Mat_{hx}")
    m.use_nodes = True
    nt = m.node_tree
    for n in list(nt.nodes): nt.nodes.remove(n)
    out = nt.nodes.new('ShaderNodeOutputMaterial')
    bsdf = nt.nodes.new('ShaderNodeBsdfPrincipled')
    bsdf.inputs["Base Color"].default_value = (rgb_lin[0], rgb_lin[1], rgb_lin[2], 1.0)
    if "Roughness" in bsdf.inputs:
        bsdf.inputs["Roughness"].default_value = 0.55
    spec = bsdf.inputs.get("Specular") or bsdf.inputs.get("Specular IOR Level")
    if spec:
        spec.default_value = 0.0
    nt.links.new(bsdf.outputs["BSDF"], out.inputs["Surface"])
    cache[key] = m
    return m

def colored_cube_mesh(base_me, rgb_lin, mesh_cache, mat_cache):
    key = (round(rgb_lin[0],4), round(rgb_lin[1],4), round(rgb_lin[2],4))
    if key in mesh_cache:
        return mesh_cache[key]
    hx = rgb_to_hex(rgb_lin)
    me = base_me.copy()
    me.name = f"Cube_{hx}"
    me.materials.clear()
    me.materials.append(make_color_material(rgb_lin, mat_cache))
    mesh_cache[key] = me
    return me

# ------------------------------------------------------------
# Fully MeshSafe texture/material color sampler
# FIX: bilinear sampling + raycast (6 dirs) priority to avoid color bleed
# ------------------------------------------------------------
def _barycentric(p, a, b, c):
    v0 = b - a; v1 = c - a; v2 = p - a
    d00 = v0.dot(v0); d01 = v0.dot(v1); d11 = v1.dot(v1)
    d20 = v2.dot(v0); d21 = v2.dot(v1)
    denom = d00 * d11 - d01 * d01
    if abs(denom) < 1e-12:
        return (1.0, 0.0, 0.0)
    v = (d11 * d20 - d01 * d21) / denom
    w = (d00 * d21 - d01 * d20) / denom
    u = 1.0 - v - w
    return (u, v, w)

def _load_image_pixels(img):
    w, h = img.size[0], img.size[1]
    return w, h, list(img.pixels)

def _sample_img_uv_linear(img_pack, uv, alpha_thr=0.001):
    w, h, px = img_pack
    u = uv[0] % 1.0
    v = uv[1] % 1.0
    x = int(u * (w-1))
    y = int((1.0 - v) * (h-1))
    i = (y*w + x) * 4
    r,g,b,a = px[i], px[i+1], px[i+2], px[i+3]
    if a < alpha_thr:
        return (0.8, 0.8, 0.8)
    return (float(r), float(g), float(b))

def _sample_img_uv_bilinear(img_pack, uv, alpha_thr=0.001):
    w, h, px = img_pack
    if w <= 1 or h <= 1:
        return _sample_img_uv_linear(img_pack, uv, alpha_thr)

    u = uv[0] % 1.0
    v = uv[1] % 1.0

    fx = u * (w - 1)
    fy = (1.0 - v) * (h - 1)

    x0 = int(math.floor(fx)); y0 = int(math.floor(fy))
    x1 = min(x0 + 1, w - 1)
    y1 = min(y0 + 1, h - 1)

    tx = fx - x0
    ty = fy - y0

    def get_rgba(ix, iy):
        i = (iy * w + ix) * 4
        return (px[i], px[i+1], px[i+2], px[i+3])

    r00,g00,b00,a00 = get_rgba(x0,y0)
    r10,g10,b10,a10 = get_rgba(x1,y0)
    r01,g01,b01,a01 = get_rgba(x0,y1)
    r11,g11,b11,a11 = get_rgba(x1,y1)

    a0 = a00*(1-tx) + a10*tx
    a1 = a01*(1-tx) + a11*tx
    a  = a0*(1-ty) + a1*ty
    if a < alpha_thr:
        return (0.8, 0.8, 0.8)

    r0 = r00*(1-tx) + r10*tx
    g0 = g00*(1-tx) + g10*tx
    b0 = b00*(1-tx) + b10*tx

    r1 = r01*(1-tx) + r11*tx
    g1 = g01*(1-tx) + g11*tx
    b1 = b01*(1-tx) + b11*tx

    r = r0*(1-ty) + r1*ty
    g = g0*(1-ty) + g1*ty
    b = b0*(1-ty) + b1*ty

    return (float(r), float(g), float(b))

def _apply_mapping_fast(uv, mapping_tuple):
    if not mapping_tuple:
        return uv
    locx, locy, rotz, scalex, scaley = mapping_tuple
    u = uv[0] * scalex + locx
    v = uv[1] * scaley + locy
    ca = math.cos(rotz); sa = math.sin(rotz)
    return (u*ca - v*sa, u*sa + v*ca)

def _find_principled(mat):
    if not mat or not mat.use_nodes or not mat.node_tree:
        return None
    for n in mat.node_tree.nodes:
        if n.type == 'BSDF_PRINCIPLED':
            return n
    return None

def _get_linked_node(socket):
    return socket.links[0].from_node if socket and socket.is_linked and socket.links else None

def _resolve_image_chain_fast(mat):
    pr = _find_principled(mat)
    if not pr:
        return (None, None, None)
    base = pr.inputs.get("Base Color")
    if not base or not base.is_linked:
        return (None, None, None)
    tex = base.links[0].from_node
    if not tex or tex.type != 'TEX_IMAGE' or not getattr(tex, "image", None):
        return (None, None, None)

    mapping_tuple = None
    uvmap_name = None
    n = _get_linked_node(tex.inputs.get("Vector"))
    if n and n.type == 'MAPPING':
        loc = n.inputs["Location"].default_value
        rot = n.inputs["Rotation"].default_value
        scl = n.inputs["Scale"].default_value
        mapping_tuple = (float(loc[0]), float(loc[1]), float(rot[2]), float(scl[0]), float(scl[1]))
        n2 = _get_linked_node(n.inputs.get("Vector"))
        if n2 and n2.type == 'UVMAP':
            uvmap_name = n2.uv_map
    elif n and n.type == 'UVMAP':
        uvmap_name = n.uv_map

    return (tex, uvmap_name, mapping_tuple)

def get_base_color_linear(mat):
    pr = _find_principled(mat)
    if pr:
        col = pr.inputs["Base Color"].default_value
        return (float(col[0]), float(col[1]), float(col[2]))
    try:
        c = mat.diffuse_color
        return (float(c[0]), float(c[1]), float(c[2]))
    except Exception:
        return (0.8, 0.8, 0.8)

class MeshSafeColorSampler:
    def __init__(self, src_obj, depsgraph, step_hint=0.0):
        self.alpha_thr = 0.001
        self.mat_cache = {}
        self.img_cache = {}
        self.step_hint = float(step_hint)
        self.max_cast = max(1e-6, self.step_hint * 2.25) if self.step_hint > 0 else None

        eval_obj = src_obj.evaluated_get(depsgraph)
        me = eval_obj.to_mesh()
        try:
            if not me.loop_triangles:
                me.calc_loop_triangles()

            mw = eval_obj.matrix_world
            verts_w = [mw @ v.co for v in me.vertices]

            tris = []
            self.tri_data = []

            default_uv = me.uv_layers.active if me.uv_layers else None

            for tri in me.loop_triangles:
                v0, v1, v2 = tri.vertices
                tris.append((v0, v1, v2))

                a = verts_w[v0]; b = verts_w[v1]; c = verts_w[v2]

                poly = me.polygons[tri.polygon_index]
                mi = poly.material_index
                mat = src_obj.material_slots[mi].material if mi < len(src_obj.material_slots) else None

                kind, payload, mapping_tuple, chosen_uv = self._material_sampler(mat, me, default_uv)

                if kind == "IMG" and chosen_uv is not None:
                    l0, l1, l2 = tri.loops
                    uv0 = (float(chosen_uv.data[l0].uv[0]), float(chosen_uv.data[l0].uv[1]))
                    uv1 = (float(chosen_uv.data[l1].uv[0]), float(chosen_uv.data[l1].uv[1]))
                    uv2 = (float(chosen_uv.data[l2].uv[0]), float(chosen_uv.data[l2].uv[1]))
                else:
                    uv0 = uv1 = uv2 = (0.0, 0.0)

                self.tri_data.append((a,b,c, uv0,uv1,uv2, kind, payload, mapping_tuple))

            self.bvh = BVHTree.FromPolygons(verts_w, tris, all_triangles=True)
            if self.bvh is None:
                raise RuntimeError("Không tạo được BVH cho sampler (FromPolygons).")
        finally:
            try: eval_obj.to_mesh_clear()
            except Exception: pass

    def _material_sampler(self, mat, me, default_uv):
        if not mat:
            return ("CONST", (0.8,0.8,0.8), None, None)

        key = mat.name_full
        if key in self.mat_cache:
            kind, payload, mapping_tuple, uv_name = self.mat_cache[key]
            uv_layer = me.uv_layers.get(uv_name) if (uv_name and me.uv_layers) else default_uv
            return kind, payload, mapping_tuple, uv_layer

        tex, uvmap_name, mapping_tuple = _resolve_image_chain_fast(mat)

        uv_layer = default_uv
        if me.uv_layers and uvmap_name and uvmap_name in me.uv_layers:
            uv_layer = me.uv_layers[uvmap_name]

        if tex and getattr(tex, "image", None) and uv_layer is not None:
            img = tex.image
            if img.name_full not in self.img_cache:
                self.img_cache[img.name_full] = _load_image_pixels(img)
            payload = self.img_cache[img.name_full]
            self.mat_cache[key] = ("IMG", payload, mapping_tuple, uvmap_name)
            return "IMG", payload, mapping_tuple, uv_layer

        payload = get_base_color_linear(mat)
        self.mat_cache[key] = ("CONST", payload, None, None)
        return "CONST", payload, None, None

    def close(self): return

    def sample_linear(self, world_point):
        eps = max(1e-6, (self.step_hint * 0.05 if self.step_hint > 0 else 1e-5))
        best = None  # (dist, loc, normal, tri_idx)

        dirs = (
            mathutils.Vector(( 1, 0, 0)),
            mathutils.Vector((-1, 0, 0)),
            mathutils.Vector(( 0, 1, 0)),
            mathutils.Vector(( 0,-1, 0)),
            mathutils.Vector(( 0, 0, 1)),
            mathutils.Vector(( 0, 0,-1)),
        )
        maxd = self.max_cast

        for d in dirs:
            origin = world_point - d * eps
            loc, normal, tri_idx, dist = self.bvh.ray_cast(origin, d, maxd if maxd else 1e30)
            if loc is None or tri_idx is None:
                continue
            if best is None or dist < best[0]:
                best = (dist, loc, normal, tri_idx)

        if best is None:
            try:
                loc, normal, tri_idx, dist = self.bvh.find_nearest(world_point, self.max_cast if self.max_cast else 0.0)
            except TypeError:
                loc, normal, tri_idx, dist = self.bvh.find_nearest(world_point)
            if tri_idx is None:
                return (0.8, 0.8, 0.8), None
        else:
            _, loc, normal, tri_idx = best

        if tri_idx < 0 or tri_idx >= len(self.tri_data):
            return (0.8, 0.8, 0.8), None

        a,b,c, uv0,uv1,uv2, kind, payload, mapping_tuple = self.tri_data[tri_idx]
        p = loc if loc is not None else world_point

        if kind == "CONST":
            return payload, normal

        u,v,w = _barycentric(p, a, b, c)
        uv = (uv0[0]*u + uv1[0]*v + uv2[0]*w, uv0[1]*u + uv1[1]*v + uv2[1]*w)
        uv = _apply_mapping_fast(uv, mapping_tuple)
        col = _sample_img_uv_bilinear(payload, uv, alpha_thr=self.alpha_thr)
        return col, normal

# ------------------------------------------------------------
# Uniform Grid Fill (NO giant cubes ever)
# Target => determines step size; always fill inside cells.
# ------------------------------------------------------------
def step_from_target(bbox, target):
    minx, miny, minz, maxx, maxy, maxz = bbox
    sx = max(1e-9, maxx-minx)
    sy = max(1e-9, maxy-miny)
    sz = max(1e-9, maxy-miny)  # (kept as-is from your code style, but harmless)
    # NOTE: keep your original vol calc correctly:
    sz = max(1e-9, maxz-minz)
    vol = sx*sy*sz
    target = max(1, int(target))
    return (vol / target) ** (1.0/3.0)

def count_inside_for_step(bvh, diag, bbox, step):
    minx, miny, minz, maxx, maxy, maxz = bbox
    sx = maxx-minx; sy=maxy-miny; sz=maxz-minz
    nx = max(1, int(math.ceil(sx / step)))
    ny = max(1, int(math.ceil(sy / step)))
    nz = max(1, int(math.ceil(sz / step)))

    half = step * 0.5
    cnt = 0
    for iz in range(nz):
        z = minz + half + iz*step
        for iy in range(ny):
            y = miny + half + iy*step
            for ix in range(nx):
                x = minx + half + ix*step
                if inside_vote(bvh, mathutils.Vector((x,y,z)), diag):
                    cnt += 1
    return cnt

def choose_step_by_target(bvh, diag, bbox, target, iters=10):
    base = step_from_target(bbox, target)
    lo = base / 6.0
    hi = base * 6.0

    best_step = base
    best_diff = 10**18

    for _ in range(iters):
        mid = (lo + hi) * 0.5
        c = count_inside_for_step(bvh, diag, bbox, mid)
        diff = abs(c - target)
        if diff < best_diff:
            best_diff = diff
            best_step = mid
        if c < target:
            hi = mid
        else:
            lo = mid

    return max(1e-9, best_step)

def uniform_grid_points(bvh, diag, bbox, step, limit_max=2000000):
    minx, miny, minz, maxx, maxy, maxz = bbox
    sx = maxx-minx; sy=maxy-miny; sz=maxz-minz
    nx = max(1, int(math.ceil(sx / step)))
    ny = max(1, int(math.ceil(sy / step)))
    nz = max(1, int(math.ceil(sz / step)))

    total = nx*ny*nz
    if total > limit_max:
        scale = (total / limit_max) ** (1.0/3.0)
        step *= scale
        nx = max(1, int(math.ceil(sx / step)))
        ny = max(1, int(math.ceil(sy / step)))
        nz = max(1, int(math.ceil(sz / step)))

    half = step * 0.5
    pts = []
    for iz in range(nz):
        z = minz + half + iz*step
        for iy in range(ny):
            y = miny + half + iy*step
            for ix in range(nx):
                x = minx + half + ix*step
                p = mathutils.Vector((x,y,z))
                if inside_vote(bvh, p, diag):
                    pts.append(p)
    return pts, step

# ------------------------------------------------------------
# Build cubes (uniform size)
# ------------------------------------------------------------
def build_cubes_uniform(col, base_me, pts, step, cube_scale, gap_ratio,
                        colorize, palette_k, seed, sampler, src_obj, diag):
    mesh_cache = {}
    mat_cache = {}

    cube_size = max(0.0005, float(step) * float(cube_scale) * (1.0 - float(gap_ratio)))

    if colorize and sampler and pts:
        cols = [sampler.sample_linear(p)[0] for p in pts]
        if palette_k and palette_k > 0:
            palette = kmeans_palette(cols, int(palette_k), seed=int(seed))
            cols = [min(palette, key=lambda pc: dist2(c, pc)) for c in cols]
    else:
        cols = [(0.8, 0.8, 0.8)] * len(pts)

    created = 0
    for p, c in zip(pts, cols):
        me = colored_cube_mesh(base_me, c, mesh_cache, mat_cache) if colorize else base_me
        ob = bpy.data.objects.new("Cube", me)

        if colorize:
            hx = rgb_to_hex(c)
            ob.name = hx
            ob.color = (c[0], c[1], c[2], 1.0)

        ob.location = p
        ob.scale = (cube_size, cube_size, cube_size)
        ob.rotation_euler = (0.0, 0.0, 0.0)

        col.objects.link(ob)
        created += 1

    return created, cube_size

# ------------------------------------------------------------
# Main runner
# ------------------------------------------------------------
def run_uniform_grid(obj, out_col, base_cube,
                     target, cube_scale, gap_ratio,
                     colorize, palette_k, seed, txt_name,
                     search_iters):
    deps = bpy.context.evaluated_depsgraph_get()
    bbox = world_bbox(obj)
    bvh, diag = build_meshsafe_bvh(obj, deps)

    step = choose_step_by_target(bvh, diag, bbox, int(target), iters=int(search_iters))
    pts, step_used = uniform_grid_points(bvh, diag, bbox, step)

    # FIX: pass step_used into sampler to stabilize raycast range + eps
    sampler = MeshSafeColorSampler(obj, deps, step_hint=step_used) if colorize else None
    try:
        created, cube_size = build_cubes_uniform(
            out_col, base_cube, pts, step_used,
            cube_scale=float(cube_scale),
            gap_ratio=float(gap_ratio),
            colorize=bool(colorize),
            palette_k=int(palette_k),
            seed=int(seed),
            sampler=sampler,
            src_obj=obj,
            diag=diag
        )

        msg = f"OK: Uniform Grid Fill | cubes={created} | target≈{int(target)} | step≈{step_used:.6f} | cube≈{cube_size:.6f}"
        if colorize:
            msg += f" | palette={'∞' if int(palette_k)==0 else int(palette_k)}"
        log_to_text(msg, txt_name)
        return msg
    finally:
        if sampler:
            sampler.close()

# ------------------------------------------------------------
# UI
# ------------------------------------------------------------
def poll_mesh_obj(self, obj): return obj and obj.type == 'MESH'

class VoxelToolProps(bpy.types.PropertyGroup):
    source_mode: bpy.props.EnumProperty(
        name="Source",
        items=[('IMPORT',"Import File","Import model từ file"),
               ('SELECT',"Use Project Object","Dùng mesh có sẵn trong project")],
        default='SELECT'
    )
    input_path: bpy.props.StringProperty(name="File", subtype='FILE_PATH', default="")
    source_object: bpy.props.PointerProperty(name="Object", type=bpy.types.Object, poll=poll_mesh_obj)

    target_cubes: bpy.props.IntProperty(
        name="Target Cubes (Density)",
        default=30000, min=10, max=5000000,
        description="Mật độ (ước lượng). Addon tự chọn step sao cho số cube gần target. Grid luôn đồng đều, không có cube to bất thường."
    )
    cube_scale: bpy.props.FloatProperty(name="Cube Scale", default=1.0, min=0.1, max=2.0)
    gap_ratio: bpy.props.FloatProperty(
        name="Gap Ratio",
        default=0.0, min=0.0, max=0.9,
        description="Chỉ làm cube nhỏ lại (tạo khe). Không ảnh hưởng grid/không tạo cube to."
    )

    search_iters: bpy.props.IntProperty(
        name="Auto Fit Iterations",
        default=8, min=2, max=16,
        description="Số vòng tìm step để count gần target hơn. Tăng = chính xác hơn nhưng chậm hơn."
    )

    colorize: bpy.props.BoolProperty(name="Colorize (Material/Texture)", default=True)
    palette_colors: bpy.props.IntProperty(name="Palette Colors", default=0, min=0, max=64)
    palette_seed: bpy.props.IntProperty(name="Seed", default=42, min=0, max=999999)

    voxel_result: bpy.props.StringProperty(name="VOXEL_RESULT", default="VOXEL_RESULT.txt")
    status: bpy.props.StringProperty(name="Status", default="Sẵn sàng", options={'SKIP_SAVE'})

class VOXEL_OT_Run(bpy.types.Operator):
    bl_idname = "voxel_generator_uniform_grid_meshsafe.run"
    bl_label = "Bắt đầu tạo"
    bl_options = {'REGISTER','UNDO'}

    def execute(self, context):
        p = context.scene.voxel_tool_props_uniform_grid_meshsafe
        txt_name = (p.voxel_result or "VOXEL_RESULT.txt").strip()
        try:
            if p.source_mode == 'IMPORT':
                if not p.input_path.strip():
                    raise ValueError("Vui lòng chọn file!")
                imported = import_any(p.input_path)
                src_obj = join_meshes(imported)
                move_to_collection(src_obj, ensure_collection("VoxelOutput"))
            else:
                src_obj = p.source_object or context.active_object
                if not src_obj or src_obj.type != 'MESH':
                    raise ValueError("Chưa chọn mesh object trong project.")
                bpy.ops.object.select_all(action='DESELECT')
                src_obj.select_set(True)
                context.view_layer.objects.active = src_obj
                bpy.ops.object.transform_apply(location=True, rotation=True, scale=True)

            cubes_col = ensure_child_collection("VoxelOutput", "VoxelCubes")
            clear_collection_objects(cubes_col)
            base_cube = ensure_cube_mesh("VoxelCubeMeshBase")

            p.status = run_uniform_grid(
                src_obj, cubes_col, base_cube,
                target=int(p.target_cubes),
                cube_scale=float(p.cube_scale),
                gap_ratio=float(p.gap_ratio),
                colorize=bool(p.colorize),
                palette_k=int(p.palette_colors),
                seed=int(p.palette_seed),
                txt_name=txt_name,
                search_iters=int(p.search_iters),
            )
            popup(p.status, "Voxel Generator", 'INFO')
            return {'FINISHED'}
        except Exception as e:
            tb = traceback.format_exc()
            err = f"[ERROR] {e}"
            log_to_text(err + "\n\nTraceback:\n" + tb, txt_name)
            p.status = "Lỗi (xem VOXEL_RESULT)"
            popup(err, "Voxel Generator Error", 'ERROR')
            self.report({'ERROR'}, str(e))
            return {'CANCELLED'}

class VOXEL_PT_Panel(bpy.types.Panel):
    bl_label = "Voxel Generator (Uniform Grid MeshSafe)"
    bl_idname = "VOXEL_PT_voxel_generator_uniform_grid_meshsafe"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Voxel Tool'

    def draw(self, context):
        layout = self.layout
        p = context.scene.voxel_tool_props_uniform_grid_meshsafe

        boxs = layout.box()
        boxs.label(text="Source")
        boxs.prop(p, "source_mode", expand=True)
        if p.source_mode == 'IMPORT':
            boxs.prop(p, "input_path")
        else:
            boxs.prop(p, "source_object")

        box = layout.box()
        box.label(text="Uniform Grid Fill (No Giant Cubes)")
        box.prop(p, "target_cubes")
        box.prop(p, "search_iters")
        box.prop(p, "cube_scale")
        box.prop(p, "gap_ratio")

        bc = layout.box()
        bc.label(text="Color")
        bc.prop(p, "colorize")
        col = bc.column()
        col.enabled = p.colorize
        col.prop(p, "palette_colors")
        col.prop(p, "palette_seed")

        layout.separator()
        layout.operator("voxel_generator_uniform_grid_meshsafe.run", icon='PLAY')

        layout.separator()
        b3 = layout.box()
        b3.label(text="VOXEL_RESULT")
        b3.prop(p, "voxel_result")
        layout.label(text=p.status, icon='INFO')

classes = (VoxelToolProps, VOXEL_OT_Run, VOXEL_PT_Panel)

def register():
    for c in classes:
        bpy.utils.register_class(c)
    bpy.types.Scene.voxel_tool_props_uniform_grid_meshsafe = bpy.props.PointerProperty(type=VoxelToolProps)

def unregister():
    if hasattr(bpy.types.Scene, "voxel_tool_props_uniform_grid_meshsafe"):
        del bpy.types.Scene.voxel_tool_props_uniform_grid_meshsafe
    for c in reversed(classes):
        bpy.utils.unregister_class(c)

if __name__ == "__main__":
    register()
